/*
 *  Driver for S3 SonicVibes soundcard
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/sonicvibes.h"
#include "../include/opl3.h"
#include "../include/initval.h"

MODULE_DESCRIPTION("\
Driver: S3 SonicVibes PCI\n\
PCI: 0x5333=0xca00\n\
");

#ifndef PCI_VENDOR_ID_S3
#define PCI_VENDOR_ID_S3             0x5333
#endif
#ifndef PCI_DEVICE_ID_S3_SONICVIBES
#define PCI_DEVICE_ID_S3_SONICVIBES  0xca00
#endif

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_dmaa_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
int snd_dmac_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
int snd_reverb[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 0};
int snd_mge[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 0};
unsigned int snd_dmaio = 0x7a00;	/* DDMA i/o address */
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for S3 SonicVibes soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for S3 SonicVibes soundcard.");
MODULE_PARM(snd_dmaa_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dmaa_size, "DDMA-A size in kB for S3 SonicVibes soundcard. [DMA_SIZE]");
MODULE_PARM(snd_dmac_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dmac_size, "DDMA-C size in kB for S3 SonicVibes soundcard. [DMA_SIZE]");
MODULE_PARM(snd_reverb, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_reverb, "Enable reverb (SRAM is present) for S3 SonicVibes soundcard. [BOOL]");
MODULE_PARM(snd_mge, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_mge, "MIC Gain Enable for S3 SonicVibes soundcard. [BOOL]");
MODULE_PARM(snd_dmaio, "i");
MODULE_PARM_DESC(snd_dmaio, "DDMA i/o base address for S3 SonicVibes soundcard. [PORT]");

struct snd_sonic {
	struct pci_dev *pci;
	snd_irq_t *irqptr;
	snd_dma_t *dma1ptr;
	snd_dma_t *dma2ptr;
	snd_card_t *card;
	sonicvibes_t *sonic;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *midi_uart;
	snd_hwdep_t *synth;
};

static struct snd_sonic *snd_sonic_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_sonic_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_sonic_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static int __init snd_sonic_detect(snd_card_t * card,
				   struct snd_sonic *scard)
{
	int idx;
	unsigned int dmaa, dmac;

	if ((scard->pci = pci_find_device(PCI_VENDOR_ID_S3,
					  PCI_DEVICE_ID_S3_SONICVIBES,
					  scard->pci)) == NULL)
		return -ENODEV;
	for (idx = 0; idx < 5; idx++) {
		if (pci_resource_start(scard->pci, idx) == 0 ||
		    !(pci_resource_flags(scard->pci, idx) & IORESOURCE_IO))
			return -ENODEV;
	}
	if (snd_register_ioport(card, pci_resource_start(scard->pci, 1), 0x10, "S3 SonicVibes PCM", NULL) < 0)
		goto __nodev;
	if (snd_register_ioport(card, pci_resource_start(scard->pci, 2), 4, "S3 SonicVibes Synth", NULL) < 0)
		goto __nodev;
	if (snd_register_ioport(card, pci_resource_start(scard->pci, 3), 4, "S3 SonicVibes Midi", NULL) < 0)
		goto __nodev;
	pci_read_config_dword(scard->pci, 0x40, &dmaa);
	pci_read_config_dword(scard->pci, 0x48, &dmac);
	snd_dmaio &= ~0x0f;
	dmaa &= ~0x0f;
	dmac &= ~0x0f;
	if (!dmaa) {
		dmaa = snd_dmaio;
		snd_dmaio += 0x10;
		snd_printk("SV: BIOS didn't allocate DDMA channel A i/o, allocated at 0x%x\n", dmaa);
	}
	if (!dmac) {
		dmac = snd_dmaio;
		snd_dmaio += 0x10;
		snd_printk("SV: BIOS didn't allocate DDMA channel C i/o, allocated at 0x%x\n", dmac);
	}
	pci_write_config_dword(scard->pci, 0x40, dmaa);
	pci_write_config_dword(scard->pci, 0x48, dmac);
	if (snd_register_ioport(card, dmaa, 0x10, "S3 SonicVibes DDMA-A", NULL) < 0)
		goto __nodev;
	if (snd_register_ioport(card, dmac, 0x10, "S3 SonicVibes DDMA-C", NULL) < 0)
		goto __nodev;
	return 0;
      __nodev:
	snd_unregister_ioports(card);
	return -ENODEV;
}

static void snd_sonic_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_sonic *scard = (struct snd_sonic *) dev_id;

	if (scard == NULL || scard->sonic == NULL)
		return;
	snd_sonicvibes_interrupt(scard->sonic);
}

static int __init snd_sonic_resources(snd_card_t * card,
				      struct snd_sonic *scard, int dev)
{
	int err;

	if ((err = snd_register_interrupt(card,
			"S3 SonicVibes", scard->pci->irq,
			SND_IRQ_TYPE_PCI, snd_sonic_interrupt,
			scard, NULL, &scard->irqptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card,
			"S3 SonicVibes DDMA-A", 0,
			SND_DMA_TYPE_PCI_16MB, snd_dmaa_size[dev],
			NULL, &scard->dma1ptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card,
			"S3 SonicVibes DDMA-C", 1,
			SND_DMA_TYPE_PCI_16MB, snd_dmac_size[dev],
			NULL, &scard->dma2ptr)) < 0)
		return err;
	return 0;
}

static int __init snd_sonic_probe(int dev, struct snd_sonic *scard)
{
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
	snd_rawmidi_t *midi_uart = NULL;
	snd_hwdep_t *synth = NULL;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_sonic_use_inc, snd_sonic_use_dec);
	if (card == NULL)
		return -ENOMEM;
	card->type = SND_CARD_TYPE_S3_SONICVIBES;
	scard->pci = NULL;
	do {
		if (!snd_sonic_detect(card, scard))
			break;
	} while (scard->pci);
	if (scard->pci == NULL) {
		snd_card_free(card);
		return -ENODEV;
	}
	if (snd_sonic_resources(card, scard, dev) < 0) {
		snd_card_free(card);
		return -ENODEV;
	}
	if (snd_sonicvibes_create(card, scard->pci,
				  scard->dma1ptr, scard->dma2ptr,
				  scard->irqptr,
				  snd_reverb[dev] ? 1 : 0,
				  snd_mge[dev] ? 1 : 0,
				  &scard->sonic) < 0)
		goto __nodev;
	if (snd_sonicvibes_pcm(scard->sonic, 0, &pcm) < 0)
		goto __nodev;
	if (snd_sonicvibes_mixer(scard->sonic, 0, pcm, &mixer) < 0)
		goto __nodev;
	if (snd_mpu401_uart_new(card, 0, MPU401_HW_SONICVIBES,
				scard->sonic->midi_port,
				scard->irqptr->irq,
				&midi_uart) < 0)
		goto __nodev;
	snd_sonicvibes_midi(scard->sonic, midi_uart);
	if (snd_opl3_new(card, 0, scard->sonic->synth_port,
			 scard->sonic->synth_port + 2,
			 OPL3_HW_OPL3_SV, -1, &synth) < 0)
		goto __nodev;
	strcpy(card->abbreviation, "SonicVibes");
	strcpy(card->shortname, "S3 SonicVibes");
	sprintf(card->longname, "%s rev %i at 0x%lx, irq %i",
		card->shortname,
		scard->sonic->revision,
		pci_resource_start(scard->pci, 1),
		scard->irqptr->irq);

	if (!snd_card_register(card)) {
		scard->card = card;
		scard->mixer = mixer;
		scard->pcm = pcm;
		scard->midi_uart = scard->sonic->rmidi = midi_uart;
		scard->synth = synth;
		return 0;
	}
	goto __nodev;

      __nodev:
	snd_card_free(card);
	return -ENXIO;
}

static int __exit snd_sonic_free(int dev)
{
	struct snd_sonic *scard;

	scard = snd_sonic_cards[dev];
	snd_sonic_cards[dev] = NULL;
	if (scard) {
		snd_card_unregister(scard->card);
		snd_kfree(scard);
	}
	return 0;
}

static int __init alsa_card_sonicvibes_init(void)
{
	int dev, cards;
	struct snd_sonic *scard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		scard = (struct snd_sonic *)
				snd_kcalloc(sizeof(struct snd_sonic), GFP_KERNEL);
		if (scard == NULL)
			continue;
		if (snd_sonic_probe(dev, scard) < 0) {
			snd_kfree(scard);
			break;
		}
		snd_sonic_cards[dev] = scard;
		cards++;
	}
	if (!cards) {
#ifdef MODULE
		snd_printk("S3 SonicVibes soundcard #%i not found or device busy\n", dev + 1);
#endif
		return -ENODEV;
	}
	return 0;
}

static void __exit alsa_card_sonicvibes_exit(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_sonic_free(dev);
}

module_init(alsa_card_sonicvibes_init)
module_exit(alsa_card_sonicvibes_exit)
