/*
 *  The driver for the Cirrus Logic's Sound Fusion CS461X based soundcards
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../include/driver.h"
#include "../include/cs461x.h"
#include "../include/initval.h"

MODULE_DESCRIPTION("\
Driver: Cirrus Logic Sound Fusion CS461X\n\
Card: Generic CS4280\n\
Card: Generic CS4610\n\
Card: Generic CS4612\n\
Card: Generic CS4614\n\
Card: Generic CS4615\n\
PCI: 0x1013=0x6001\n\
PCI: 0x1013=0x6003\n\
PCI: 0x1013=0x6004\n\
");

int snd_index[SND_CARDS] = SND_DEFAULT_IDX;	/* Index 0-MAX */
char *snd_id[SND_CARDS] = SND_DEFAULT_STR;	/* ID for this card */
int snd_dac_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
int snd_adc_frame_size[SND_CARDS] = {[0 ... (SND_CARDS - 1)] = 128};
MODULE_PARM(snd_index, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_index, "Index value for the CS461x soundcard.");
MODULE_PARM(snd_id, "1-" __MODULE_STRING(SND_CARDS) "s");
MODULE_PARM_DESC(snd_id, "ID string for the CS461x soundcard.");
MODULE_PARM(snd_dac_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_dac_frame_size, "DAC frame size in kB for the CS461x soundcard.");
MODULE_PARM(snd_adc_frame_size, "1-" __MODULE_STRING(SND_CARDS) "i");
MODULE_PARM_DESC(snd_adc_frame_size, "ADC frame size in kB for the CS461x soundcard.");

struct snd_cs461x {
	struct pci_dev *pci;
	snd_irq_t *irqptr;
	snd_dma_t *dma1ptr;	/* DAC frame */
	snd_dma_t *dma2ptr;	/* ADC frame */
	snd_card_t *card;
	cs461x_t *codec;
	snd_pcm_t *pcm;
	snd_kmixer_t *mixer;
	snd_rawmidi_t *rmidi;
};

static struct snd_cs461x *snd_cs461x_cards[SND_CARDS] = SND_DEFAULT_PTR;

static void snd_card_cs461x_use_inc(snd_card_t * card)
{
	MOD_INC_USE_COUNT;
}

static void snd_card_cs461x_use_dec(snd_card_t * card)
{
	MOD_DEC_USE_COUNT;
}

static int __init snd_card_cs461x_detect(snd_card_t * card,
					 struct snd_cs461x *scard,
					 unsigned short device)
{
	int idx;

	if ((scard->pci = pci_find_device(PCI_VENDOR_ID_CIRRUS,
					  device,
					  scard->pci)) == NULL)
		return -ENODEV;
	for (idx = 0; idx < SND_CARDS; idx++) {
		if (snd_cards[idx]) {
			if (pci_resource_start(snd_cs461x_cards[idx]->pci, 0) == pci_resource_start(scard->pci, 0))
				return -EBUSY;
		}
	}
	return 0;
}

static void snd_card_cs461x_interrupt(int irq, void *dev_id, struct pt_regs *regs)
{
	struct snd_cs461x *scard = (struct snd_cs461x *) dev_id;

	if (scard == NULL || scard->codec == NULL)
		return;
	snd_cs461x_interrupt(scard->codec);
}

static int __init snd_card_cs461x_resources(snd_card_t * card,
					    struct snd_cs461x *scard, int dev)
{
	int err;

	if ((err = snd_register_interrupt(card,
			"CS461x", scard->pci->irq,
			SND_IRQ_TYPE_PCI, snd_card_cs461x_interrupt,
			scard, NULL, &scard->irqptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card,
			"CS461x - DAC frame", 0,
			SND_DMA_TYPE_PCI, snd_dac_frame_size[dev],
			NULL, &scard->dma1ptr)) < 0)
		return err;
	if ((err = snd_register_dma_channel(card,
			"CS461x - ADC frame", 2,
			SND_DMA_TYPE_PCI, snd_adc_frame_size[dev],
			NULL, &scard->dma2ptr)) < 0)
		return err;
	return 0;
}

static int __init snd_card_cs461x_probe(int dev, struct snd_cs461x *scard)
{
	snd_card_t *card;
	snd_pcm_t *pcm = NULL;
	snd_kmixer_t *mixer = NULL;
	snd_rawmidi_t *rmidi = NULL;

	card = snd_card_new(snd_index[dev], snd_id[dev],
			    snd_card_cs461x_use_inc, snd_card_cs461x_use_dec);
	if (card == NULL)
		return -ENOMEM;
	card->type = SND_CARD_TYPE_CS461X;
	scard->pci = NULL;
	do {
		if (!snd_card_cs461x_detect(card, scard, PCI_DEVICE_ID_CIRRUS_4610))
			break;
	} while (scard->pci);
	if (scard->pci == NULL) {
		do {
			if (!snd_card_cs461x_detect(card, scard, PCI_DEVICE_ID_CIRRUS_4612))
				break;
		} while (scard->pci);
	}			
	if (scard->pci == NULL) {
		do {
			if (!snd_card_cs461x_detect(card, scard, PCI_DEVICE_ID_CIRRUS_4615))
				break;
		} while (scard->pci);
	}			
	if (scard->pci == NULL) {
		snd_card_free(card);
		return -ENODEV;
	}
	if (snd_card_cs461x_resources(card, scard, dev) < 0) {
		snd_card_free(card);
		return -ENODEV;
	}
	if (snd_cs461x_create(card, scard->pci,
			      scard->dma1ptr,
			      scard->dma2ptr,
			      scard->irqptr,
			      &scard->codec) < 0)
		goto __nodev;
	if (snd_cs461x_pcm(scard->codec, 0, &pcm) < 0)
		goto __nodev;
	if (snd_cs461x_mixer(scard->codec, 0, pcm, &mixer) < 0)
		goto __nodev;
	if (snd_cs461x_midi(scard->codec, 0, &rmidi) < 0)
		goto __nodev;
	strcpy(card->abbreviation, "CS461x");
	strcpy(card->shortname, "Sound Fusion CS461x");
	sprintf(card->longname, "%s at 0x%lx/0x%lx, irq %i",
		card->shortname,
		pci_resource_start(scard->pci, 0),
		pci_resource_start(scard->pci, 1),
		scard->irqptr->irq);

	if (!snd_card_register(card)) {
		scard->card = card;
		scard->mixer = mixer;
		scard->pcm = pcm;
		scard->rmidi = rmidi;
		return 0;
	}
	goto __nodev;

      __nodev:
	snd_card_free(card);
	return -ENXIO;
}

#ifdef MODULE

static int __exit snd_card_cs461x_free(int dev)
{
	struct snd_cs461x *scard;

	scard = snd_cs461x_cards[dev];
	snd_cs461x_cards[dev] = NULL;
	if (scard) {
		snd_card_unregister(scard->card);
		snd_kfree(scard);
	}
	return 0;
}

#endif

static int __init alsa_card_cs461x_init(void)
{
	int dev, cards;
	struct snd_cs461x *scard;

	for (dev = cards = 0; dev < SND_CARDS; dev++) {
		scard = (struct snd_cs461x *)
				snd_kcalloc(sizeof(struct snd_cs461x), GFP_KERNEL);
		if (scard == NULL)
			continue;
		if (snd_card_cs461x_probe(dev, scard) < 0) {
			snd_kfree(scard);
			break;
		}
		snd_cs461x_cards[dev] = scard;
		cards++;
	}
	if (!cards) {
#ifdef MODULE
		snd_printk("Sound Fusion CS461x soundcard #%i not found or device busy\n", dev + 1);
#endif
		return -ENODEV;
	}
	return 0;
}

static void __exit alsa_card_cs461x_exit(void)
{
	int dev;

	for (dev = 0; dev < SND_CARDS; dev++)
		snd_card_cs461x_free(dev);
}

module_init(alsa_card_cs461x_init)
module_exit(alsa_card_cs461x_exit)
