/*
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *  Routines for control of MPU-401 in UART mode
 *
 *  MPU-401 supports UART mode which is not capable generate transmit
 *  interrupts thus output is done via polling. Also, if irq < 0, then
 *  input is done also via polling. Do not expect good performance.
 *
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 *
 */

#define SND_MAIN_OBJECT_FILE
#include "../../include/driver.h"
#include "../../include/mpu401.h"

static void snd_mpu401_uart_input_read(snd_rawmidi_t * rmidi, mpu401_t * mpu);
static void snd_mpu401_uart_output_write(snd_rawmidi_t * rmidi, mpu401_t * mpu);

/*

 */

void snd_mpu401_uart_interrupt(snd_rawmidi_t * rmidi)
{
	mpu401_t *mpu;

	mpu = snd_magic_cast(mpu401_t, rmidi->private_data, );
	if (mpu == NULL)
		return;
	if (mpu->mode & MPU401_MODE_INPUT)
		snd_mpu401_uart_input_read(rmidi, mpu);
	/* ok. for better Tx performance try do some output when input is done */
	if (mpu->mode & MPU401_MODE_OUTPUT)
		snd_mpu401_uart_output_write(rmidi, mpu);
}

static void snd_mpu401_uart_timer(unsigned long data)
{
	unsigned long flags;
	mpu401_t *mpu = snd_magic_cast(mpu401_t, (void *)data, );

	spin_lock_irqsave(&mpu->timer_lock, flags);
	mpu->mode |= MPU401_MODE_TIMER;
	mpu->timer.expires = 1 + jiffies;
	add_timer(&mpu->timer);
	spin_unlock_irqrestore(&mpu->timer_lock, flags);
	if (mpu->rmidi) {
		if (mpu->mode & MPU401_MODE_INPUT)
			snd_mpu401_uart_input_read(mpu->rmidi, mpu);
		if (mpu->mode & MPU401_MODE_OUTPUT)
			snd_mpu401_uart_output_write(mpu->rmidi, mpu);
	}
}

static void snd_mpu401_uart_add_timer (mpu401_t *mpu, int cond)
{
	unsigned long flags;

	spin_lock_irqsave (&mpu->timer_lock, flags);
	if ((mpu->mode & MPU401_MODE_TIMER) == 0 && cond) {
		mpu->timer.data = (unsigned long)mpu;
		mpu->timer.function = snd_mpu401_uart_timer;
		mpu->timer.expires = 1 + jiffies;
		add_timer(&mpu->timer);
		mpu->mode |= MPU401_MODE_OUTPUT_TIMER;
	} 
	spin_unlock_irqrestore (&mpu->timer_lock, flags);
}

static void snd_mpu401_uart_remove_timer (mpu401_t *mpu)
{
	unsigned long flags;

	spin_lock_irqsave (&mpu->timer_lock, flags);
	if (mpu->mode & MPU401_MODE_TIMER) {
		mpu->mode &= ~MPU401_MODE_TIMER;
		del_timer(&mpu->timer);
	}
	spin_unlock_irqrestore (&mpu->timer_lock, flags);
}

/*

 */

static void snd_mpu401_uart_cmd(mpu401_t * mpu, unsigned char cmd, int ack)
{
	unsigned long flags;
	int timeout, ok;

	spin_lock_irqsave(&mpu->input_lock, flags);
	if (mpu->hardware != MPU401_HW_TRID4DWAVE) {
		outb(0x00, MPU401D(mpu));
		for (timeout = 100000; timeout > 0 && !(inb(MPU401C(mpu)) & 0x80); timeout--)
			inb(MPU401D(mpu));
#ifdef CONFIG_SND_DEBUG
		if (timeout <= 0)
			snd_printk("snd_mpu401_uart_cmd: clear rx timeout (status = 0x%x)\n", inb(MPU401C(mpu)));
#endif
	}
	/* ok. standard MPU-401 initialization */
	if (mpu->hardware != MPU401_HW_SB) {
		for (timeout = 1000; timeout > 0 && inb(MPU401C(mpu)) & 0x40; timeout--)
			udelay(10);
#ifdef CONFIG_SND_DEBUG
		if (!timeout)
			snd_printk("snd_mpu401_uart_cmd: tx timeout (status = 0x%x)\n", inb(MPU401C(mpu)));
#endif
	}
	outb(cmd, MPU401C(mpu));
	if (ack) {
		ok = 0;
		timeout = 10000;
		while (!ok && timeout-- > 0) {
			if (!(inb(MPU401C(mpu)) & 0x80)) {
				if (inb(MPU401D(mpu)) == 0xfe)
					ok = 1;
			}
		}
	} else {
		ok = 1;
	}
	spin_unlock_irqrestore(&mpu->input_lock, flags);
	if (!ok) {
		snd_printk("snd_mpu401_uart_cmd: 0x%x failed at 0x%x (status = 0x%x, data = 0x%x)\n", cmd, mpu->port, inb(MPU401C(mpu)), inb(MPU401D(mpu)));
	}
	// snd_printk("snd_mpu401_uart_cmd: 0x%x at 0x%x (status = 0x%x, data = 0x%x)\n", cmd, mpu->port, inb(MPU401C(mpu)), inb(MPU401D(mpu)));
}

static int snd_mpu401_uart_input_open(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	mpu401_t *mpu;

	mpu = snd_magic_cast(mpu401_t, rmidi->private_data, -ENXIO);
	// snd_printk("[0x%x] MPU-401 command port - 0x%x\n", MPU401C(mpu), inb(MPU401C(mpu)));
	// snd_printk("[0x%x] MPU-401 data port - 0x%x\n", MPU401D(mpu), inb(MPU401D(mpu)));
	spin_lock_irqsave(&mpu->open_lock, flags);
	mpu->mode |= MPU401_MODE_INPUT;
	if (mpu->open_input)
		mpu->open_input(mpu);
	if (!(mpu->mode & MPU401_MODE_OUTPUT)) {
		spin_unlock_irqrestore(&mpu->open_lock, flags);
		snd_mpu401_uart_cmd(mpu, 0xff, 1);	/* reset */
		snd_mpu401_uart_cmd(mpu, 0x3f, 1);	/* enter UART mode */
	} else {
		spin_unlock_irqrestore(&mpu->open_lock, flags);
	}
	return 0;
}

static int snd_mpu401_uart_output_open(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	mpu401_t *mpu;

	mpu = snd_magic_cast(mpu401_t, rmidi->private_data, -ENXIO);
	spin_lock_irqsave(&mpu->open_lock, flags);
	mpu->mode |= MPU401_MODE_OUTPUT;
	if (mpu->open_output)
		mpu->open_output(mpu);
	if (!(mpu->mode & MPU401_MODE_INPUT)) {
		spin_unlock_irqrestore(&mpu->open_lock, flags);
		snd_mpu401_uart_cmd(mpu, 0xff, 1);	/* reset */
		snd_mpu401_uart_cmd(mpu, 0x3f, 1);	/* enter UART mode */
	} else {
		spin_unlock_irqrestore(&mpu->open_lock, flags);
	}
	return 0;
}

static int snd_mpu401_uart_input_close(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	mpu401_t *mpu;

	mpu = snd_magic_cast(mpu401_t, rmidi->private_data, -ENXIO);
	spin_lock_irqsave(&mpu->open_lock, flags);
	mpu->mode &= ~MPU401_MODE_INPUT;
	if (mpu->close_input)
		mpu->close_input(mpu);
	if (!(mpu->mode & MPU401_MODE_OUTPUT)) {
		spin_unlock_irqrestore(&mpu->open_lock, flags);
		snd_mpu401_uart_cmd(mpu, 0xff, 0);	/* reset */
	} else {
		spin_unlock_irqrestore(&mpu->open_lock, flags);
	}
	return 0;
}

static int snd_mpu401_uart_output_close(snd_rawmidi_t * rmidi)
{
	unsigned long flags;
	mpu401_t *mpu;

	mpu = snd_magic_cast(mpu401_t, rmidi->private_data, -ENXIO);
	spin_lock_irqsave(&mpu->open_lock, flags);
	if (mpu->close_output)
		mpu->close_output(mpu);
	mpu->mode &= ~MPU401_MODE_OUTPUT;
	if (!(mpu->mode & MPU401_MODE_INPUT)) {
		spin_unlock_irqrestore(&mpu->open_lock, flags);
		snd_mpu401_uart_cmd(mpu, 0xff, 0);	/* reset */
	} else {
		spin_unlock_irqrestore(&mpu->open_lock, flags);
	}
	return 0;
}

static void snd_mpu401_uart_input_trigger(snd_rawmidi_t * rmidi, int up)
{
	unsigned long flags;
	mpu401_t *mpu;
	int max = 64;

	mpu = snd_magic_cast(mpu401_t, rmidi->private_data, );
	spin_lock_irqsave(&mpu->input_lock, flags);
	if (up) {
		if ((mpu->mode & MPU401_MODE_INPUT_TRIGGER) == 0) {
			mpu->mode |= MPU401_MODE_INPUT_TRIGGER;
			/* flush FIFO */
			while (max-- > 0)
				inb(MPU401D(mpu));
		}
		snd_mpu401_uart_add_timer(mpu, mpu->irq < 0);
		mpu->mode |= MPU401_MODE_INPUT_TRIGGER;
	} else {
		snd_mpu401_uart_remove_timer (mpu);
		mpu->mode &= ~(MPU401_MODE_INPUT_TRIGGER | MPU401_MODE_INPUT_TIMER);
	}
	spin_unlock_irqrestore(&mpu->input_lock, flags);
	if (up)
		snd_mpu401_uart_input_read(rmidi, mpu);
}

static void snd_mpu401_uart_input_read(snd_rawmidi_t * rmidi, mpu401_t * mpu)
{
	unsigned long flags;
	int max = 128, mode;
	unsigned char byte, status;

	while (max-- > 0) {
		spin_lock_irqsave(&mpu->input_lock, flags);
		status = inb(MPU401C(mpu));
		if (status == 0xff) {
			spin_unlock_irqrestore(&mpu->input_lock, flags);
			break;	/* wrong value, MPU isn't connected */
		}
		if (!(status & 0x80)) {
			byte = inb(MPU401D(mpu));
			mode = mpu->mode & MPU401_MODE_INPUT_TRIGGER;
			spin_unlock_irqrestore(&mpu->input_lock, flags);
			if (mode)
				snd_rawmidi_receive(rmidi, &byte, 1);
		} else {
			spin_unlock_irqrestore(&mpu->input_lock, flags);
			break;
		}
	}
}

/*
 *  Tx FIFO sizes:
 *    CS4237B			- 16 bytes
 *    AudioDrive ES1688         - 12 bytes
 *    S3 SonicVibes             -  8 bytes
 *    SoundBlaster AWE 64       -  2 bytes (ugly hardware)
 */

static void snd_mpu401_uart_output_write(snd_rawmidi_t * rmidi, mpu401_t * mpu)
{
	unsigned long flags;
	unsigned char status, byte;
	int max = 256, timeout;

	while (max > 0) {
		for (timeout = 100; timeout > 0; timeout--) {
			if ((inb(MPU401C(mpu)) & 0x40) == 0)
				break;
		}
		spin_lock_irqsave(&mpu->output_lock, flags);
		status = inb(MPU401C(mpu));
		if ((status & 0x40) == 0) {
			if (mpu->mode & MPU401_MODE_OUTPUT) {
				if (snd_rawmidi_transmit(rmidi, &byte, 1) == 1) {
					outb(byte, MPU401D(mpu));
					max--;
				} else {
					snd_mpu401_uart_remove_timer (mpu);
					status = 0x40;	/* return */
				}
			} else {
				status = 0x40;	/* return */
			}
		}
		spin_unlock_irqrestore(&mpu->output_lock, flags);
		if (status & 0x40)
			break;
	}
}

static void snd_mpu401_uart_output_trigger(snd_rawmidi_t * rmidi, int up)
{
	unsigned long flags;
	mpu401_t *mpu;

	mpu = snd_magic_cast(mpu401_t, rmidi->private_data, );
	spin_lock_irqsave(&mpu->output_lock, flags);
	if (up) {
		snd_mpu401_uart_add_timer(mpu, 1);
		mpu->mode |= MPU401_MODE_OUTPUT_TRIGGER;
	} else {
		snd_mpu401_uart_remove_timer(mpu);
		mpu->mode &= ~(MPU401_MODE_OUTPUT_TRIGGER | MPU401_MODE_OUTPUT_TIMER);
	}
	spin_unlock_irqrestore(&mpu->output_lock, flags);
	if (up)
		snd_mpu401_uart_output_write(rmidi, mpu);
}

/*

 */

static struct snd_stru_rawmidi_channel_hw snd_mpu401_uart_output =
{
	open:		snd_mpu401_uart_output_open,
	close:		snd_mpu401_uart_output_close,
	trigger:	snd_mpu401_uart_output_trigger,
};

static struct snd_stru_rawmidi_channel_hw snd_mpu401_uart_input =
{
	open:		snd_mpu401_uart_input_open,
	close:		snd_mpu401_uart_input_close,
	trigger:	snd_mpu401_uart_input_trigger,
};

static void snd_mpu401_uart_free(void *private_data)
{
	mpu401_t *mpu = snd_magic_cast(mpu401_t, private_data, );
	snd_magic_kfree(mpu);
}

int snd_mpu401_uart_new(snd_card_t * card, int device,
			unsigned short hardware,
			int port,
			int irq,
			snd_rawmidi_t ** rrawmidi)
{
	mpu401_t *mpu;
	snd_rawmidi_t *rmidi;
	int err;

	*rrawmidi = NULL;
	if ((err = snd_rawmidi_new(card, "MPU-401U", device, &rmidi)) < 0)
		return err;
	mpu = snd_magic_kcalloc(mpu401_t, 0, GFP_KERNEL);
	if (mpu == NULL) {
		snd_device_free(card, rmidi);
		return -ENOMEM;
	}
	spin_lock_init(&mpu->open_lock);
	spin_lock_init(&mpu->input_lock);
	spin_lock_init(&mpu->output_lock);
	spin_lock_init(&mpu->timer_lock);
	mpu->hardware = hardware;
	mpu->port = port;
	mpu->irq = irq;
	strcpy(rmidi->name, "MPU-401 (UART)");
	memcpy(&rmidi->chn[SND_RAWMIDI_CHANNEL_OUTPUT].hw, &snd_mpu401_uart_output, sizeof(snd_mpu401_uart_output));
	memcpy(&rmidi->chn[SND_RAWMIDI_CHANNEL_INPUT].hw, &snd_mpu401_uart_input, sizeof(snd_mpu401_uart_input));
	rmidi->info_flags |= SND_RAWMIDI_INFO_OUTPUT |
	                     SND_RAWMIDI_INFO_INPUT |
	                     SND_RAWMIDI_INFO_DUPLEX;
	rmidi->private_data = mpu;
	rmidi->private_free = snd_mpu401_uart_free;
	*rrawmidi = mpu->rmidi = rmidi;
	return 0;
}

EXPORT_SYMBOL(snd_mpu401_uart_interrupt);
EXPORT_SYMBOL(snd_mpu401_uart_new);

/*
 *  INIT part
 */

static int __init alsa_mpu401_uart_init(void)
{
	return 0;
}

static void __exit alsa_mpu401_uart_exit(void)
{
}

module_init(alsa_mpu401_uart_init)
module_exit(alsa_mpu401_uart_exit)
